import hashlib
import math
import os

# ----------------------------
# HDGL Movie Encoder Stub
# ----------------------------

FIBS = [1,1,2,3,5,8,13,21,34,55,89,144,233,377,610]  # Fibonacci for delta
MAX_SEED = 2**128 - 1  # arbitrary seed size for covenant

def file_to_hash(filepath):
    """Compute SHA256 of file as unique movie ID."""
    sha256 = hashlib.sha256()
    with open(filepath, "rb") as f:
        while True:
            chunk = f.read(4096)
            if not chunk:
                break
            sha256.update(chunk)
    return sha256.hexdigest()

def generate_seed(filepath):
    """Produce deterministic seed from file hash."""
    movie_hash = file_to_hash(filepath)
    # Convert first 16 bytes to integer as seed
    seed_int = int(movie_hash[:32], 16) % MAX_SEED
    return seed_int

def generate_delta(seed, index=0):
    """Produce a small delta using Fibonacci sequence for uniqueness."""
    return FIBS[index % len(FIBS)]

def encode_movie(filepath, fib_index=0):
    """Return HDGL covenant snippet for a movie file."""
    movie_id = file_to_hash(filepath)
    seed = generate_seed(filepath)
    delta = generate_delta(seed, fib_index)
    
    snippet = (
        f"0x{movie_id[:32]} 0x{seed:032x} 0x{delta:032x} OP_HDGL_VEC_ADD OP_EQUALVERIFY"
    )
    return snippet

# ----------------------------
# Example usage
# ----------------------------
if __name__ == "__main__":
    movie_file = "example_movie.mp4"
    
    if not os.path.exists(movie_file):
        print(f"Error: {movie_file} not found!")
    else:
        covenant_snippet = encode_movie(movie_file)
        print("=== HDGL Covenant Snippet ===")
        print(covenant_snippet)
